using System;
using System.IO;
using System.Collections;
using System.Reflection;
using System.Xml;
using System.Xml.Schema;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class validates VistALink messages XML representation. It will accept XML string,
	/// XSD file name and perform the validation. All XSD files used by this class must be
	/// stored as embedded resources in the assembly (subfolder XSDs).
	/// </summary>
	public sealed class VistALinkMessageXmlVsXsdValidator
	{
		private static readonly Hashtable _schemas = new Hashtable( 20 );
		private static readonly Assembly _executingAssembly = Assembly.GetExecutingAssembly();

		private VistALinkMessageXmlVsXsdValidator() {}
		static VistALinkMessageXmlVsXsdValidator() {}	// get rid of "beforefieldinit" attribute (DON'T REMOVE!)

		/// <summary>
		/// Will validate given XML string against the XSD schema 
		/// with a specified name stored as embedded resource. 
		/// Since this method is for assertive use only, it will throw an exception
		/// if a given XML cannot be validated. 
		/// </summary>
		/// <param name="xsdFileName">XSD schema file name.</param>
		/// <param name="xmlDocumentString">VistALink message XML string to validate.</param>
		public static void ValidateXmlStringVsVistALinkMessageXsd( string xsdFileName, string xmlDocumentString )
		{
			if( xsdFileName == null )
				throw( new ArgumentNullException( "xsdFileName" ) );

			if( xmlDocumentString == null )
				throw( new ArgumentNullException( "xmlDocumentString" ) );

            //VBECS 2.0 CVS .Net framwork 4.0 does not support validating reader class, recoded to use settings and xml reader.
			XmlReader _rdr = GetXmlValidatingReaderBoundToVistAlinkMessageXsd( xsdFileName, xmlDocumentString );

			try
			{
				while( _rdr.Read() );
			}
			catch( XmlSchemaException xcp )
			{
				throw( new VistALinkMessageXmlVsXsdValidationException( SR.Exceptions.VistALinkMessageXmlCannotBeXsdValidated( xsdFileName ), xcp ) );
			}
		}

		/// <summary>
		/// Lazy load for XSD schema files stored in assembly resources. Loads any given schema 
		/// from subfolder XSDs on first request. 
		/// </summary>
        /// 
		/// <param name="xsdFileName">XSD schema file name.</param>
		/// <returns>Reference to XSD schema.</returns>
		private static XmlSchema GetXsdSchemaByFileName( string xsdFileName )
		{
			if( _schemas.ContainsKey( xsdFileName ) )
				return (XmlSchema)_schemas[xsdFileName];

			try
			{
				Stream _xsdFileStream = _executingAssembly.GetManifestResourceStream( "gov.va.med.vbecs.DAL.VistALink.OpenLibrary.XSDs." + xsdFileName );

				if( _xsdFileStream == null )
					throw( new VistALinkException( SR.Exceptions.UnableToFindXSDSchemaInResources( xsdFileName ) ) );

				XmlSchema _result = XmlSchema.Read( _xsdFileStream, null );
				_schemas.Add( xsdFileName, _result );

				return _result;
			}
			catch( XmlSchemaException xcp )
			{
				throw( new VistALinkException( SR.Exceptions.UnableToLoadXSDSchema(), xcp ) );
			}
		}

		/// <summary>
		/// Returns XmlReader open on given XML fragment and bound to a specified XSD 
		/// stored in assembly resources. 
		/// </summary>
		/// <param name="xsdFileName">XSD schema file name.</param>
		/// <param name="xmlDocumentString">VistALink message XML string to validate.</param>
		/// <returns>Open on a given XML fragment and bound to a specified XSD XmlReader.</returns>
		public static XmlReader GetXmlValidatingReaderBoundToVistAlinkMessageXsd( string xsdFileName, string xmlDocumentString )
		{
            //VBECS 2.0 CVS .Net framwork 4.0 does not support validating reader class, recoded to use settings and xml reader.
			if( xsdFileName == null )
				throw( new ArgumentNullException( "xsdFileName" ) );

			if( xmlDocumentString == null )
				throw( new ArgumentNullException( "xmlDocumentString" ) );
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.ConformanceLevel = ConformanceLevel.Document;
            settings.IgnoreWhitespace = true;
            settings.ValidationType = ValidationType.Schema;
            settings.Schemas.Add(GetXsdSchemaByFileName(xsdFileName));
            XmlReader _rdr = XmlReader.Create(xmlDocumentString, settings);
            return _rdr;
		}
	}
}
